////////////////////////////////////////////////////////////////////////////////
//
// Copyright (c) since 1999. VASCO DATA SECURITY
// All rights reserved. http://www.vasco.com
//
////////////////////////////////////////////////////////////////////////////////

#import <Foundation/Foundation.h>
#import <MSSOrchestration/OrchestrationUser.h>
#import <MSSOrchestration/RemoteTransactionDisplayDataCallerDelegate.h>
NS_ASSUME_NONNULL_BEGIN

@class PasswordError;

/**
 * <p>The {@link RemoteTransactionDelegate} interface defines the specific elements required for the
 * integrator to interact during the Remote Transaction Validation flow.</p>
 * <p>This flow is initiated by the server; it is started by a call to {@link OrchestrationProtocol#execute:(NSString *)}
 * with the command provided by the server.</p>
 * <p>A {@link RemoteTransactionDelegate} object should be provided to the {@link Orchestrator} object (See {@link OrchestrationProtocol#setRemoteTransactionDelegate(RemoteTransactionDelegate)})
 * in order to receive feedback during the remote transaction validation process.</p>
 */
NS_REFINED_FOR_SWIFT
@protocol RemoteTransactionDelegate <NSObject>

/**
 * Called when the Orchestration SDK needs the {@link RemoteTransactionDelegate} object to display data
 * to the user for approval.
 *
 * @param dataToDisplay the data to display
 * @param caller        a {@link RemoteTransactionDisplayDataCallerDelegate} object to use upon user's response
 * @param user the user currently being authenticated
 */
- (void)onRemoteTransactionForOrchestrationUser:(OrchestrationUser *)user dataToDisplay:(NSString *)dataToDisplay caller:(id <RemoteTransactionDisplayDataCallerDelegate>)caller;

/**
 * Called when a step of the remote transaction validation process is complete. The provided orchestration
 * command is to be sent to the server.
 *
 * @param command an orchestration command that must be sent to the server
 */
- (void)onRemoteTransactionStepCompleteWithCommand:(NSString *)command;


/**
 * The {@link RemoteTransactionSuccessSessionState} lists possible session states after a remote transaction success.
 */
NS_REFINED_FOR_SWIFT
typedef enum {
    RemoteTransactionSuccessSessionStateAccepted,
    RemoteTransactionSuccessSessionStateRefused
} RemoteTransactionSuccessSessionState;

/**
 * Called upon remote transaction validation success
 *
 * @param state the end state of the session.
 */
- (void)onRemoteTransactionSuccessWithState:(RemoteTransactionSuccessSessionState)state;

/**
 * The {@link RemoteTransactionSessionOutdatedReason} lists possible reasons for a session to be outdated.
 */
NS_REFINED_FOR_SWIFT
typedef enum {
    RemoteTransactionSessionExpired,
    RemoteTransactionSessionAccepted,
    RemoteTransactionSessionRefused,
    RemoteTransactionSessionUnknown
} RemoteTransactionSessionOutdatedReason;

/**
 * Called upon remote transaction session outdated
 */
- (void)onRemoteTransactionSessionOutdated:(RemoteTransactionSessionOutdatedReason)reason;

/**
 * Called upon remote transaction validation abortion
 */
- (void)onRemoteTransactionAborted;

/**
 * Called when an error occurred during the remote transaction process
 *
 * @param error password error
 */
- (void)onRemoteTransactionPasswordError:(PasswordError *)error;

@end

NS_ASSUME_NONNULL_END
